<?php
// wallet_lib.php
// فانکشن‌های اصلی مدیریت کیف پول
// واحد همه مبلغ‌ها: ریال

declare(strict_types=1);

/**
 * گرفتن موجودی فعلی کیف پول کاربر
 */
function wallet_get_balance(PDO $pdo, int $userId): int
{
    $stmt = $pdo->prepare('SELECT wallet_balance FROM users WHERE id = :id LIMIT 1');
    $stmt->execute([':id' => $userId]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        return 0;
    }
    return (int)$row['wallet_balance'];
}

/**
 * ثبت تراکنش کیف پول + به‌روزرسانی موجودی
 *
 * amount:
 *   >0  شارژ (charge / refund / adjust+)
 *   <0  خرج (spend / adjust-)
 *
 * type: charge, spend, refund, adjust
 */
function wallet_add_transaction(
    PDO $pdo,
    int $userId,
    int $amount,
    string $type,
    ?int $relatedPaymentId = null,
    ?int $relatedServiceId = null,
    ?string $description = null
): bool {
    if ($amount === 0) {
        return false;
    }

    // تراکنش DB
    $pdo->beginTransaction();
    try {
        // گرفتن موجودی فعلی
        $stmtUser = $pdo->prepare('SELECT wallet_balance FROM users WHERE id = :id FOR UPDATE');
        $stmtUser->execute([':id' => $userId]);
        $userRow = $stmtUser->fetch(PDO::FETCH_ASSOC);
        if (!$userRow) {
            $pdo->rollBack();
            return false;
        }

        $currentBalance = (int)$userRow['wallet_balance'];
        $newBalance     = $currentBalance + $amount;

        // جلوگیری از منفی شدن (برای spend)
        if ($newBalance < 0) {
            $pdo->rollBack();
            return false;
        }

        // درج رکورد در wallet_transactions
        $stmtTx = $pdo->prepare(
            'INSERT INTO wallet_transactions
               (user_id, amount, type, related_payment_id, related_service_id, description, created_at)
             VALUES
               (:user_id, :amount, :type, :pay_id, :svc_id, :desc, NOW())'
        );
        $stmtTx->execute([
            ':user_id' => $userId,
            ':amount'  => $amount,
            ':type'    => $type,
            ':pay_id'  => $relatedPaymentId,
            ':svc_id'  => $relatedServiceId,
            ':desc'    => $description,
        ]);

        // آپدیت موجودی در users
        $stmtUpd = $pdo->prepare(
            'UPDATE users
             SET wallet_balance = :balance
             WHERE id = :id'
        );
        $stmtUpd->execute([
            ':balance' => $newBalance,
            ':id'      => $userId,
        ]);

        $pdo->commit();
        return true;
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        // error_log($e->getMessage());
        return false;
    }
}

/**
 * خرج کردن از کیف پول برای خرید سرویس
 * amount باید مبلغ پلن به ریال و مقدار مثبت باشد.
 * این تابع خودش amount را منفی می‌کند.
 */
function wallet_spend_for_service(
    PDO $pdo,
    int $userId,
    int $amount,
    ?int $serviceId = null,
    ?int $paymentId = null,
    ?string $description = null
): bool {
    if ($amount <= 0) {
        return false;
    }
    // amount منفی برای spend
    $amount = -1 * $amount;

    return wallet_add_transaction(
        $pdo,
        $userId,
        $amount,
        'spend',
        $paymentId,
        $serviceId,
        $description
    );
}

/**
 * شارژ کیف پول بعد از پرداخت موفق درگاه
 * amount باید مثبت و بر اساس ریال باشد.
 */
function wallet_charge_from_gateway(
    PDO $pdo,
    int $userId,
    int $amount,
    ?int $paymentId = null,
    ?string $description = null
): bool {
    if ($amount <= 0) {
        return false;
    }

    return wallet_add_transaction(
        $pdo,
        $userId,
        $amount,
        'charge',
        $paymentId,
        null,
        $description
    );
}
