<?php
declare(strict_types=1);

session_start();
require __DIR__ . '/db.php';
require __DIR__ . '/push_notifications.php'; // برای diminet_push_to_user

header('Content-Type: application/json; charset=utf-8');

/**
 * تبدیل created_at به متن تاریخ شمسی (اگر intl موجود باشد)،
 * وگرنه فرمت معمولی میلادی.
 */
function formatJalali(?string $datetime): string
{
    if (empty($datetime)) {
        return '';
    }

    $ts = strtotime($datetime);
    if ($ts === false) {
        return '';
    }

    // اگر intl فعال باشد و تقویم Persian پشتیبانی شود
    if (class_exists('IntlDateFormatter')) {
        try {
            static $fmt = null;
            if ($fmt === null) {
                $fmt = new IntlDateFormatter(
                    'fa_IR@calendar=persian', // لوکال + تقویم شمسی
                    IntlDateFormatter::MEDIUM,
                    IntlDateFormatter::SHORT,
                    'Asia/Tehran',
                    IntlDateFormatter::TRADITIONAL,
                    'yyyy/MM/dd HH:mm'
                );
            }

            $out = $fmt->format($ts);
            if ($out !== false) {
                return (string)$out;
            }
        } catch (Throwable $e) {
            // اگر intl خطا داد، می‌افتیم روی fallback پایین
        }
    }

    // فallback: تاریخ میلادی عادی
    return date('Y/m/d H:i', $ts);
}

/**
 * پیدا کردن user_id از روی ایمیل سشن
 */
function current_user_id(PDO $pdo): ?int
{
    $email = $_SESSION['user_email'] ?? '';
    if ($email === '') {
        return null;
    }

    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = :e LIMIT 1");
    $stmt->execute([':e' => $email]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    return $row ? (int)$row['id'] : null;
}

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$userId = current_user_id($pdo);

if (!$userId) {
    http_response_code(401);
    echo json_encode(
        ['ok' => false, 'error' => 'not_logged_in'],
        JSON_UNESCAPED_UNICODE
    );
    exit;
}

if ($method === 'GET') {
    $status = $_GET['status'] ?? 'all'; // all | unread
    $limit  = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;

    if ($limit < 1)   $limit = 1;
    if ($limit > 200) $limit = 200;

    $where = "user_id = :uid";
    if ($status === 'unread') {
        $where .= " AND is_read = 0";
    }

    $sql = "
        SELECT
            id,
            type,
            title,
            body,
            related_service_id,
            related_refund_id,
            is_read,
            created_at,
            pushed_at
        FROM notifications
        WHERE $where
        ORDER BY created_at DESC, id DESC
        LIMIT :limit
    ";

    try {
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':uid', $userId, PDO::PARAM_INT);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();

        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Throwable $e) {
        http_response_code(500);
        echo json_encode(
            ['ok' => false, 'error' => 'db_error'],
            JSON_UNESCAPED_UNICODE
        );
        exit;
    }

    $items       = [];
    $toPushIds   = [];  // idهای نوتیف که باید پوش بگیرند

    foreach ($rows as $r) {
        $createdAt = $r['created_at'] ?? null;
        $createdFa = formatJalali($createdAt);

        $id       = (int)($r['id'] ?? 0);
        $isRead   = (int)($r['is_read'] ?? 0);
        $pushedAt = $r['pushed_at'] ?? null;

        $title = (string)($r['title'] ?? '');
        $body  = (string)($r['body'] ?? '');

        $items[] = [
            'id'         => $id,
            'type'       => (string)($r['type'] ?? ''),
            'title'      => $title,
            'body'       => $body,

            // برای مچ شدن با app.js
            'is_read'        => $isRead,
            'created_at'     => $createdAt,
            'created_at_fa'  => $createdFa,

            // meta مثل قبل
            'meta' => [
                'related_service_id' =>
                    $r['related_service_id'] !== null
                        ? (int)$r['related_service_id']
                        : null,
                'related_refund_id'  =>
                    $r['related_refund_id'] !== null
                        ? (int)$r['related_refund_id']
                        : null,
            ],
        ];

        // هر نوتیفی که:
        //  - خوانده نشده (is_read = 0)
        //  - هنوز پوش نشده (pushed_at IS NULL)
        if ($id > 0 && $isRead === 0 && empty($pushedAt)) {
            $toPushIds[] = $id;

            // پوش برای همین یوزر:
            // (اگر خیلی زیاد شد، بعداً می‌تونیم تجمیعش کنیم)
            try {
                $payload = [
                    'title' => $title !== '' ? $title : 'اعلان جدید دیمینت',
                    'body'  => $body,
                    // مثلا بفرستش سمت پنل / تب اعلان‌ها
                    'url'   => '/index.php?tab=panel',
                ];
                diminet_push_to_user($pdo, $userId, $payload);
            } catch (Throwable $e) {
                // خطای پوش نباید API رو بترکونه
            }
        }
    }

    // حالا برای همه idهایی که پوش فرستادیم، pushed_at را ست کن
    if (!empty($toPushIds)) {
        $toPushIds = array_values(array_unique($toPushIds));
        $placeholders = implode(',', array_fill(0, count($toPushIds), '?'));
        $sqlUpdate = "
            UPDATE notifications
            SET pushed_at = NOW()
            WHERE user_id = ?
              AND id IN ($placeholders)
        ";
        try {
            $stmtU   = $pdo->prepare($sqlUpdate);
            $params  = array_merge([$userId], $toPushIds);
            $stmtU->execute($params);
        } catch (Throwable $e) {
            // اگر نتونستیم pushed_at رو ست کنیم، مهم نیست. نهایتاً دفعه بعد دوباره تلاش می‌کنه.
        }
    }

    echo json_encode(
        [
            'ok'            => true,
            'notifications' => $items,
        ],
        JSON_UNESCAPED_UNICODE
    );
    exit;
}

// ========== POST: مارک‌کردن به عنوان خوانده‌شده مثل قبل ==========

if ($method === 'POST') {
    // مارک‌کردن نوتیف‌ها به عنوان خوانده‌شده
    $raw  = file_get_contents('php://input');
    $data = json_decode($raw, true);

    if (!is_array($data)) {
        http_response_code(400);
        echo json_encode(
            ['ok' => false, 'error' => 'invalid_json'],
            JSON_UNESCAPED_UNICODE
        );
        exit;
    }

    $ids = $data['ids'] ?? [];
    if (!is_array($ids) || !$ids) {
        http_response_code(400);
        echo json_encode(
            ['ok' => false, 'error' => 'empty_ids'],
            JSON_UNESCAPED_UNICODE
        );
        exit;
    }

    $idsInt = [];
    foreach ($ids as $id) {
        $i = (int)$id;
        if ($i > 0) {
            $idsInt[] = $i;
        }
    }
    $idsInt = array_values(array_unique($idsInt));

    if (!$idsInt) {
        http_response_code(400);
        echo json_encode(
            ['ok' => false, 'error' => 'empty_ids'],
            JSON_UNESCAPED_UNICODE
        );
        exit;
    }

    $placeholders = implode(',', array_fill(0, count($idsInt), '?'));
    $sql = "
        UPDATE notifications
        SET
            is_read = 1
        WHERE
            user_id = ?
            AND id IN ($placeholders)
    ";

    try {
        $stmt   = $pdo->prepare($sql);
        $params = array_merge([$userId], $idsInt);
        $stmt->execute($params);
    } catch (Throwable $e) {
        http_response_code(500);
        echo json_encode(
            ['ok' => false, 'error' => 'db_error'],
            JSON_UNESCAPED_UNICODE
        );
        exit;
    }

    echo json_encode(
        [
            'ok'      => true,
            'updated' => $stmt->rowCount(),
        ],
        JSON_UNESCAPED_UNICODE
    );
    exit;
}

http_response_code(405);
echo json_encode(
    ['ok' => false, 'error' => 'method_not_allowed'],
    JSON_UNESCAPED_UNICODE
);
