<?php
// api/create_payment.php
// API برای ساخت رکورد پرداخت و گرفتن لینک پرداخت زرین‌پال

header('Content-Type: application/json; charset=utf-8');

require __DIR__ . '/../db.php';
require __DIR__ . '/../settings.php';

// اجازه بدیم از جاوااسکریپت فرانت (همون دامنه) صداش کنن
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// هندل preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['ok' => false, 'error' => 'Method Not Allowed']);
    exit;
}

// ورودی‌ها رو از JSON بگیر
$raw = file_get_contents('php://input');
$data = json_decode($raw, true);
if (!is_array($data)) {
    $data = $_POST; // اگر JSON نبود، از فرم معمولی
}

$email          = trim($data['email'] ?? '');
$planCode       = trim($data['plan'] ?? '');
$renewUsername  = trim($data['renew_username'] ?? ''); // برای تمدید، خالی = خرید جدید

if ($email === '' || $planCode === '') {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'ایمیل و کد پلن الزامی است.']);
    exit;
}

// پلن رو از settings بگیر
$plan = get_plan($planCode);
if (!$plan) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'پلن نامعتبر است.']);
    exit;
}

$amountRial = (int)($plan['price_rial'] ?? 0);
$isFree     = ($amountRial === 0);

try {
    // اگر پلن رایگانه → بدون زرین‌پال، مستقیم رکورد paid بساز و callback رو برگردون
    if ($isFree) {

        $stmt = $pdo->prepare("
            INSERT INTO payments (email, amount_rial, plan, status, authority, renew_username)
            VALUES (:email, :amount, :plan, 'paid', '', :renew)
        ");
        $stmt->execute([
            ':email' => $email,
            ':amount'=> $amountRial,
            ':plan'  => $planCode,
            ':renew' => $renewUsername !== '' ? $renewUsername : null,
        ]);

        $pid = (int)$pdo->lastInsertId();

        $redirectUrl = '/callback.php?free=1&pid=' . $pid;

        echo json_encode([
            'ok'           => true,
            'mode'         => 'free',
            'redirect_url' => $redirectUrl,
        ]);
        exit;
    }

    // پلن پولی → تماس با زرین‌پال
    $merchantId  = get_setting('zarinpal_merchant_id');
    $callbackUrl = get_setting('zarinpal_callback_url');

    if (!$merchantId || !$callbackUrl) {
        http_response_code(500);
        echo json_encode(['ok' => false, 'error' => 'تنظیمات پرداخت ناقص است.']);
        exit;
    }

    $description = $plan['title'] ?? ('خرید VPN - ' . $planCode);

    $payload = [
        'merchant_id'  => $merchantId,
        'amount'       => $amountRial,
        'callback_url' => $callbackUrl,
        'description'  => $description,
        'metadata'     => [
            'email' => $email,
        ],
    ];

    $ch = curl_init('https://api.zarinpal.com/pg/v4/payment/request.json');
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'Accept: application/json',
        ],
        CURLOPT_POSTFIELDS     => json_encode($payload, JSON_UNESCAPED_UNICODE),
        CURLOPT_TIMEOUT        => 30,
    ]);

    $respBody = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlErr  = curl_error($ch);
    curl_close($ch);

    if ($httpCode !== 200) {
        error_log("ZP_REQUEST_HTTP=$httpCode ERR=$curlErr RESP=$respBody");
        http_response_code(502);
        echo json_encode(['ok' => false, 'error' => 'خطا در ارتباط با درگاه پرداخت.']);
        exit;
    }

    $resp = json_decode($respBody, true);

    if (!isset($resp['data']['code']) || (int)$resp['data']['code'] !== 100) {
        $msg = $resp['errors']['message'] ?? 'خطا در درخواست پرداخت.';
        error_log("ZP_REQUEST_FAIL resp=$respBody");
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => $msg]);
        exit;
    }

    $authority = $resp['data']['authority'] ?? null;
    if (!$authority) {
        http_response_code(500);
        echo json_encode(['ok' => false, 'error' => 'authority از درگاه دریافت نشد.']);
        exit;
    }

    // ثبت رکورد پرداخت در DB با وضعیت pending
    $stmt = $pdo->prepare("
        INSERT INTO payments (email, amount_rial, plan, status, authority, renew_username)
        VALUES (:email, :amount, :plan, 'pending', :auth, :renew)
    ");
    $stmt->execute([
        ':email' => $email,
        ':amount'=> $amountRial,
        ':plan'  => $planCode,
        ':auth'  => $authority,
        ':renew' => $renewUsername !== '' ? $renewUsername : null,
    ]);

    // لینک صفحه پرداخت
    $paymentUrl = 'https://www.zarinpal.com/pg/StartPay/' . $authority;

    echo json_encode([
        'ok'          => true,
        'mode'        => 'paid',
        'payment_url' => $paymentUrl,
    ]);
    exit;

} catch (Throwable $e) {
    error_log('CREATE_PAYMENT_ERROR: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['ok' => false, 'error' => 'خطای داخلی سرور.']);
    exit;
}
