<?php
// api_service_usage.php
// خروجی: history مصرف یک سرویس از جدول service_usage_logs
// احراز هویت بر اساس ایمیل (مثل panel_data.php)

declare(strict_types=1);
ini_set('display_errors', '0');
error_reporting(E_ALL);
date_default_timezone_set('UTC');

session_start();
header('Content-Type: application/json; charset=utf-8');

// 1) احراز هویت مثل panel_data.php
if (empty($_SESSION['user_email'])) {
    http_response_code(401);
    echo json_encode([
        'ok'      => false,
        'error'   => 'not_logged_in',
        'message' => 'ابتدا وارد شوید.',
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

require __DIR__ . '/db.php';

if (!isset($pdo) || !($pdo instanceof PDO)) {
    http_response_code(500);
    echo json_encode([
        'ok'    => false,
        'error' => 'db_not_ready',
    ]);
    exit;
}

$email = $_SESSION['user_email'] ?? '';

$serviceId = isset($_GET['service_id']) ? (int)$_GET['service_id'] : 0;
if ($serviceId <= 0) {
    http_response_code(400);
    echo json_encode([
        'ok'    => false,
        'error' => 'invalid_service_id',
    ]);
    exit;
}

// مثلا فقط آخرین 200 رکورد (برای سبک شدن)
$limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 200;
if ($limit <= 0 || $limit > 1000) {
    $limit = 200;
}

try {
    // 2) مثل panel_data.php: پیدا کردن یوزر از روی ایمیل
    $stmtUser = $pdo->prepare("
        SELECT id, email
        FROM users
        WHERE email = :e
        LIMIT 1
    ");
    $stmtUser->execute([':e' => $email]);
    $user = $stmtUser->fetch(PDO::FETCH_ASSOC) ?: null;

    if (!$user) {
        http_response_code(404);
        echo json_encode([
            'ok'      => false,
            'error'   => 'user_not_found',
            'message' => 'کاربر یافت نشد.',
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        exit;
    }

    $currentUserId = (int)$user['id'];

    // 3) چک کن سرویس برای همین یوزر است
    $stmtSvc = $pdo->prepare("
        SELECT id, user_id, data_limit_bytes
        FROM services
        WHERE id = :sid
        LIMIT 1
    ");
    $stmtSvc->execute([':sid' => $serviceId]);
    $svc = $stmtSvc->fetch(PDO::FETCH_ASSOC);

    if (!$svc) {
        http_response_code(404);
        echo json_encode([
            'ok'    => false,
            'error' => 'service_not_found',
        ]);
        exit;
    }

    if ((int)$svc['user_id'] !== $currentUserId) {
        http_response_code(403);
        echo json_encode([
            'ok'    => false,
            'error' => 'forbidden',
        ]);
        exit;
    }

    // 4) لاگ‌های مصرف این سرویس
    // طبق ساختار جدول: snapshot_time, used_bytes, up_bytes, down_bytes, data_limit_bytes, status
    $sqlLog = "
        SELECT
            snapshot_time AS snapshot_at,
            status,
            data_limit_bytes,
            used_bytes,
            up_bytes,
            down_bytes
        FROM service_usage_logs
        WHERE service_id = :sid
        ORDER BY snapshot_time ASC
        LIMIT {$limit}
    ";

    $stmtLog = $pdo->prepare($sqlLog);
    $stmtLog->execute([':sid' => $serviceId]);
    $rows = $stmtLog->fetchAll(PDO::FETCH_ASSOC) ?: [];

    echo json_encode([
        'ok'      => true,
        'service' => [
            'id'               => (int)$svc['id'],
            'data_limit_bytes' => $svc['data_limit_bytes'] === null
                ? null
                : (int)$svc['data_limit_bytes'],
        ],
        'items'   => array_map(function ($r) {
            return [
                // از alias استفاده می‌کنیم
                'snapshot_at'      => $r['snapshot_at'] ?? null,
                'status'           => $r['status'] ?? null,
                'data_limit_bytes' => isset($r['data_limit_bytes']) ? (int)$r['data_limit_bytes'] : null,
                'used_bytes'       => isset($r['used_bytes']) ? (int)$r['used_bytes'] : null,
                'up_bytes'         => isset($r['up_bytes']) ? (int)$r['up_bytes'] : null,
                'down_bytes'       => isset($r['down_bytes']) ? (int)$r['down_bytes'] : null,
            ];
        }, $rows),
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Throwable $e) {
    // برای دیباگ
    error_log('[API][service_usage] ' . $e->getMessage());

    http_response_code(500);
    echo json_encode([
        'ok'           => false,
        'error'        => 'server_error',
        'error_detail' => $e->getMessage(),   // بعداً می‌تونی برداریش
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}
