<?php
declare(strict_types=1);

ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
error_reporting(E_ALL);

session_start();
require __DIR__ . '/db.php';
require __DIR__ . '/push_notifications.php'; // برای diminet_push_broadcast

header('Content-Type: application/json; charset=utf-8');

// 0 = مهمان، 1 = لاگین
$isLogged = !empty($_SESSION['user_email']) ? 1 : 0;

/**
 * تبدیل created_at به متن تاریخ شمسی (اگر intl موجود باشد)،
 * وگرنه فرمت معمولی میلادی.
 */
function formatJalali(?string $datetime): string
{
    if (empty($datetime)) {
        return '';
    }

    $ts = strtotime($datetime);
    if ($ts === false) {
        return '';
    }

    // اگر intl فعال باشد و تقویم Persian پشتیبانی شود
    if (class_exists('IntlDateFormatter')) {
        try {
            static $fmt = null;
            if ($fmt === null) {
                $fmt = new IntlDateFormatter(
                    'fa_IR@calendar=persian', // لوکال + تقویم شمسی
                    IntlDateFormatter::MEDIUM,
                    IntlDateFormatter::SHORT,
                    'Asia/Tehran',
                    IntlDateFormatter::TRADITIONAL,
                    'yyyy/MM/dd HH:mm'
                );
            }

            $out = $fmt->format($ts);
            if ($out !== false) {
                return (string) $out;
            }
        } catch (Throwable $e) {
            // اگر intl خطا داد، می‌افتیم روی fallback پایین
        }
    }

    // فallback: تاریخ میلادی عادی
    return date('Y/m/d H:i', $ts);
}

// ---- Query اصلی اعلان‌ها؛ pushed_at هم می‌گیریم ----

$sql = "
    SELECT
        id,
        title,
        body,
        visible_scope,
        is_active,
        created_at,
        pushed_at
    FROM announcements
    WHERE
        is_active = 1
        AND (
            visible_scope = 0
            OR (visible_scope = 1 AND :is_logged = 1)
        )
    ORDER BY id DESC
    LIMIT 10
";

try {
    $stmt = $pdo->prepare($sql);
    $stmt->execute([':is_logged' => $isLogged]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(
        [
            'ok'    => false,
            'error' => 'db_error',
        ],
        JSON_UNESCAPED_UNICODE
    );
    exit;
}

$items       = [];
$rowsForPush = []; // ردیف‌هایی که باید پوش بگیرند (pushed_at IS NULL)

foreach ($rows as $r) {
    $createdAtRaw = $r['created_at'] ?? null;
    $createdAtFa  = formatJalali($createdAtRaw);

    $id      = (int)($r['id'] ?? 0);
    $title   = (string)($r['title'] ?? '');
    $body    = (string)($r['body'] ?? '');
    $scope   = (int)($r['visible_scope'] ?? 0);
    $active  = (int)($r['is_active'] ?? 0);
    $pushedAt = $r['pushed_at'] ?? null;

    $items[] = [
        'id'            => $id,
        'title'         => $title,
        'body'          => $body,
        'level'         => 'info',
        'visible_scope' => $scope,
        // برای مچ شدن با app.js
        'created_at'    => $createdAtRaw,
        'created_at_fa' => $createdAtFa,
    ];

    // هر اعلان فعال که pushed_at خالی دارد → یک بار پوش بگیرد
    if ($id > 0 && $active === 1 && empty($pushedAt)) {
        $rowsForPush[] = $r;
    }
}

// الان JSON اصلی رو آماده می‌کنیم
$response = [
    'ok'            => true,
    'is_logged_in'  => (bool) $isLogged,
    'announcements' => $items,
];

// ======= پوش خودکار برای اعلان‌های جدید =======
if (!empty($rowsForPush)) {
    $idsToMark = [];

    foreach ($rowsForPush as $row) {
        $id    = (int)($row['id'] ?? 0);
        $title = trim((string)($row['title'] ?? ''));
        $body  = trim((string)($row['body'] ?? ''));
        $scope = (int)($row['visible_scope'] ?? 0);

        if ($id <= 0) {
            continue;
        }

        $idsToMark[] = $id;

        if ($title === '') {
            $title = 'اطلاعیه جدید دیمینت';
        }

        // اگر visible_scope = 1 باشد → فقط سابسکریپشن‌های دارای user_id
        $onlyLoggedIn = ($scope === 1);

        // آدرسی که با کلیک روی نوتیف باز می‌شود
        $url = '/index.php'; // اگر خواستی بعداً ?tab=home هم اضافه کن

        try {
            diminet_push_broadcast(
                $pdo,
                [
                    'title' => $title,
                    'body'  => $body,
                    'url'   => $url,
                    // 'icon' => '/favicon.ico',
                ],
                $onlyLoggedIn,
                1000
            );
        } catch (Throwable $e) {
            // اگر پوش این یکی خطا داد، بقیه را ادامه می‌دهیم
            continue;
        }
    }

    // بعد از تلاش برای ارسال پوش، pushed_at را بروزرسانی می‌کنیم
    if (!empty($idsToMark)) {
        $idsToMark = array_values(array_unique($idsToMark));
        $placeholders = implode(',', array_fill(0, count($idsToMark), '?'));

        $sqlUpdate = "
            UPDATE announcements
            SET pushed_at = NOW()
            WHERE id IN ($placeholders)
        ";

        try {
            $stmtU = $pdo->prepare($sqlUpdate);
            $stmtU->execute($idsToMark);
        } catch (Throwable $e) {
            // اگر نتونستیم pushed_at رو ست کنیم، مهم نیست؛ نهایتاً دفعه بعد دوباره تلاش می‌کنه.
        }
    }
}

echo json_encode($response, JSON_UNESCAPED_UNICODE);
